/*
 * @(#)RelationalERModel.java  1.0  2006-01-18
 *
 * Copyright (c) 2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm.rer.model;

import ch.hslu.cm.simulation.*;
import ch.randelshofer.util.*;
import java.util.*;
import org.jhotdraw.util.ResourceBundleUtil;

/**
 * RelationalERModel.
 *
 *
 * @author Werner Randelshofer
 * @version 1.0 2006-01-18 Created.
 */
public class RelationalERModel extends AbstractSimulation {
    public final static ResourceBundleUtil labels =
            ResourceBundleUtil.getBundle("ch.hslu.cm.rer.Labels", Locale.getDefault());
    /**
     * Enumeration of simulated concepts.
     */
    public final static int ENTITY = 0;
    public final static int RELATIONSHIP = 1;
    
    /** Creates a new instance. */
    public RelationalERModel() {
    }
    
    public Entity getSimulatedEntity(String name) {
        // FIXME - Linear search!!
        // This is very slow, we should consider using hash maps for all
        // names
        
        for (Iterator i = getElements().iterator(); i.hasNext(); ) {
            SimulatedElement elem = (SimulatedElement) i.next();
            if (elem.getSimulatedConcept() == ENTITY) {
                Entity set = (Entity) elem;
                if (set.getName().equals(name)) {
                    return set;
                }
            }
        }
        
        return null;
    }
    
    public Collection<Entity> getSimulatedEntities() {
        LinkedList<Entity> result = new LinkedList<Entity>();
        
        for (SimulatedElement elem : getElements()) {
            if (elem.getSimulatedConcept() == ENTITY) {
                result.add((Entity) elem);
            }
        }
        
        Collections.sort(result, new EntityNameComparator());
        
        return result;
    }
    public Collection<Relationship> getSimulatedRelationships() {
        LinkedList<Relationship> result = new LinkedList<Relationship>();
        
        for (SimulatedRelationship elem : getRelationships()) {
            if (elem.getSimulatedConcept() == RELATIONSHIP) {
                result.add((Relationship) elem);
            }
        }
        return result;
    }
    
    public SimulatedObject create(int simulatedConcept) {
        SimulatedObject elem;
        switch (simulatedConcept) {
            case ENTITY :
                elem = new Entity();
                break;
            case RELATIONSHIP :
                elem = new Relationship();
                break;
            default :
                throw new IllegalArgumentException(simulatedConcept+"");
        }
        return elem;
    }
    
    /**
     * Returns null, if this RelationalERModel is equivalent to the specified RelationalERModel.
     * Returns a list of Strings describing the difference to the specified RelationalERModel.
     * Returns an empty list, if the two models are equivalent.
     */
    public List<String> describeDifferencesTo(RelationalERModel that) {
        // Quickly return null when attempting to compare with ourselves.
        if (that == this) return null;
        
        // Find matching entities
        // ----------------------
        // We put all entities for which we haven't found a mapping yet
        // into a bag. If we find a mapping, we remove the entity from the bag.
        // The bag is a set, because an entity can occur only once in a model.
        HashSet<Entity> thatEntityBag = new HashSet<Entity>(that.getSimulatedEntities());
        HashSet<Entity> thisEntityBag = new HashSet<Entity>(this.getSimulatedEntities());
        
        // Mappings between this model and that model.
        // For each mapping we found, we put an entry into the map of which
        // the key is in this model, and the value is in that model.
        HashMap<Entity,Entity> entityMap = new HashMap<Entity,Entity>();
        HashMap<RERAttribute,RERAttribute> attributeMap = new HashMap<RERAttribute,RERAttribute>();
        LinkedList<String> differences = new LinkedList<String>();
        
        // Report differences in entity count.
        if (thisEntityBag.size() != thatEntityBag.size()) {
            differences.add(labels.getFormatted("diffEntityCount", thisEntityBag.size(), thatEntityBag.size()));
        }
        
        // Map entities by name.
        // A matching of name plus the number of attributes and number of
        // relationships has a higher quality than a matching of only the name.
        for (int quality = 3; quality > 0; quality--) {
            int match;
            do {
                match = 0;
                entityMatcher: for (Entity thisE : thisEntityBag) {
                    for (Entity thatE : thatEntityBag) {
                        match = 0;
                        if (thisE.getName().toLowerCase().equals(thatE.getName().toLowerCase())) {
                            match++;
                            if (thisE.getRelationships().size() == thatE.getRelationships().size()) {
                                match++;
                            }
                            if (thisE.getAttributeCount() == thatE.getAttributeCount()) {
                                match++;
                            }
                        }
                        if (match == quality) {
                            entityMap.put(thisE, thatE);
                            thisEntityBag.remove(thisE);
                            thatEntityBag.remove(thatE);
                            break entityMatcher;
                        }
                    }
                }
            } while (match == quality && thisEntityBag.size()  > 0 && thatEntityBag.size() > 0);
        }
        
        // Map entities by their relationships
        // A matching of all relationships has a higher quality than a matching
        // of all relationships except for one.
        for (int quality = 2; quality > 0; quality--) {
            int match;
            do {
                match = 0;
                entityRelationshipMatcher: for (Entity thisE : thisEntityBag) {
                    Collection<Entity> thisRE = thisE.getRelatedEntities();
                    if (thisRE.size() > 0 && entityMap.keySet().containsAll(thisRE)) {
                        for (Entity thatE : thatEntityBag) {
                            match = 0;
                            LinkedList<Entity> thatREBag = new LinkedList<Entity>(thatE.getRelatedEntities());
                            if (thatREBag.size() == thisRE.size() && entityMap.values().containsAll(thatREBag)) {
                                for (Entity e : thisRE) {
                                    thatREBag.remove(entityMap.get(e));
                                }
                                if (thatREBag.size() == 0) {
                                    match = 2;
                                } else if (thatREBag.size() == 1) {
                                    match = 1;
                                }
                                
                                if (match == quality) {
                                    entityMap.put(thisE, thatE);
                                    thisEntityBag.remove(thisE);
                                    thatEntityBag.remove(thatE);
                                    break entityRelationshipMatcher;
                                }
                            }
                        }
                    }
                }
            } while (match == quality && thisEntityBag.size()  > 0 && thatEntityBag.size() > 0);
        }
        
        // Report differences between entities
        // -----------------------------------
        // Report which entities are too much/too few
        for (Entity e : thisEntityBag) {
            differences.add(labels.getFormatted("diffEntityTooMuch", e.getName()));
        }
        for (Entity e : thatEntityBag) {
            differences.add(labels.getFormatted("diffEntityMissing", e.getName()));
        }
        
        // Report differences in entity properties
        for (Map.Entry<Entity, Entity> entry : entityMap.entrySet()) {
            Entity thisE = entry.getKey();
            Entity thatE = entry.getValue();
            if (! thisE.getName().toLowerCase().equals(thatE.getName().toLowerCase())) {
                differences.add(labels.getFormatted("diffEntityName", thisE.getName(), thatE.getName()));
            }
            if (thisE.getType() != thatE.getType()) {
                differences.add(labels.getFormatted("diffEntityType", thisE.getName(), thatE.getType()));
            }
        }
        
        // Map attributes
        for (Map.Entry<Entity, Entity> entry : entityMap.entrySet()) {
            Entity thisE = entry.getKey();
            Entity thatE = entry.getValue();
            
            HashSet<RERAttribute> thisEAttrBag = new HashSet<RERAttribute>(thisE.getAttributes());
            HashSet<RERAttribute> thatEAttrBag = new HashSet<RERAttribute>(thatE.getAttributes());
            
            for (int quality = 3; quality > 0; quality--) {
                int match;
                do {
                    match = 0;
                    attrMatcher: for (RERAttribute thisEA : thisEAttrBag) {
                        for (RERAttribute thatEA : thatEAttrBag) {
                            match = 0;
                            if (thisEA.getName().toLowerCase().equals(thatEA.getName().toLowerCase())) {
                                match++;
                                
                                if (thisEA.isPrimaryKey() == thatEA.isPrimaryKey()) {
                                    match++;
                                }
                                if (thisEA.isForeignKey() == thatEA.isForeignKey()) {
                                    match++;
                                }
                            } else {
                                if (thisEA.isPrimaryKey() && thatEA.isPrimaryKey()) {
                                    match++;
                                }
                                if (thisEA.isForeignKey() && thatEA.isForeignKey()) {
                                    match++;
                                }
                                
                            }
                            if (match == quality) {
                                attributeMap.put(thisEA, thatEA);
                                thisEAttrBag.remove(thisEA);
                                thatEAttrBag.remove(thatEA);
                                break attrMatcher;
                            }
                        }
                    }
                } while (match == quality && thisEAttrBag.size() > 0 && thatEAttrBag.size() > 0);
            }
            
            // Report differences between attributes
            // -----------------------------------
            // Report which attributes are too much/too few
            for (RERAttribute a : thisEAttrBag) {
                differences.add(labels.getFormatted("diffAttributeTooMuch", thisE.getName(), a.getName()));
            }
            for (RERAttribute a : thatEAttrBag) {
                differences.add(labels.getFormatted("diffAttributeMissing", thisE.getName(), a.getName()));
            }
            for (RERAttribute thisEA : thisE.getAttributes()) {
                RERAttribute thatEA = attributeMap.get(thisEA);
                if (thatEA != null) {
                    if (! thisEA.getName().toLowerCase().equals(thatEA.getName().toLowerCase())) {
                        differences.add(labels.getFormatted("diffAttributeName", thisE.getName(), thisEA.getName(), thatEA.getName()));
                    }
                    if (thisEA.isPrimaryKey() != thatEA.isPrimaryKey()) {
                        differences.add(labels.getFormatted(
                                (thisEA.isPrimaryKey()) ? "diffAttributeNoPrimaryKey" : "diffAttributePrimaryKey",
                                thisE.getName(), thisEA.getName())
                                );
                    }
                    if (thisEA.isForeignKey() != thatEA.isForeignKey()) {
                        differences.add(labels.getFormatted(
                                (thisEA.isForeignKey()) ? "diffAttributeNoForeignKey" : "diffAttributeForeignKey",
                                thisE.getName(), thisEA.getName())
                                );
                    }
                }
            }
        }
        
        
        // Find matching relationships
        // ---------------------------
        HashMap<Relationship,Relationship> relationshipMap = new HashMap<Relationship,Relationship>();
        HashSet<Relationship> thatRelationshipBag = new HashSet<Relationship>(that.getSimulatedRelationships());
        HashSet<Relationship> thisRelationshipBag = new HashSet<Relationship>(this.getSimulatedRelationships());
        if (thisRelationshipBag.size() != thatRelationshipBag.size()) {
            differences.add(labels.getFormatted("diffRelationshipCount", thisRelationshipBag.size(), thatRelationshipBag.size()));
        }
        
        // Throw out relationships for which we can't map the entities
        for (int quality = 1; quality > 0; quality--) {
            int match;
            do {
                match = 0;
                fullRelationshipMapper: for (Relationship thisR : thisRelationshipBag) {
                    for (Relationship thatR : thatRelationshipBag) {
                        if (entityMap.get(thisR.getStart()) == null ||
                                entityMap.get(thisR.getEnd()) == null) {
                            thisRelationshipBag.remove(thisR);
                            match = 1;
                            differences.add(labels.getFormatted("diffRelationshipNotMappable", thisR.getStart().getName(), thisR.getEnd().getName()));
                        }
                    }
                }
            } while (match == quality && thisRelationshipBag.size()  > 0 && thatRelationshipBag.size() > 0);
        }
        
        // Map relationships by matching associated entities,
        // attribute name and cardinality.
        for (int quality = 5; quality > 0; quality--) {
            int match;
            do {
                match = 0;
                relationshipMapper: for (Relationship thisR : thisRelationshipBag) {
                    for (Relationship thatR : thatRelationshipBag) {
                        match = 0;
                        if (entityMap.get(thisR.getStart()) == thatR.getStart() &&
                                entityMap.get(thisR.getEnd()) == thatR.getEnd() ||
                                entityMap.get(thisR.getStart()) == thatR.getEnd() &&
                                entityMap.get(thisR.getEnd()) == thatR.getStart()) {
                            
                            match++;
                            boolean isReverse = entityMap.get(thisR.getStart()) == thatR.getEnd();
                            
                            if (thisR.getStartLabel() != null && thatR.getStartLabel() != null &&
                                    thisR.getStartLabel().toLowerCase().equals(thatR.getStartLabel().toLowerCase())) {
                                match++;
                            }
                            
                            
                            if (thisR.getStartCardinality() == ((isReverse) ? thatR.getEndCardinality() : thatR.getStartCardinality()) &&
                                    thisR.getEndCardinality() == ((isReverse) ? thatR.getStartCardinality() : thatR.getEndCardinality())
                                    ) {
                                match++;
                            }
                            
                            RERAttribute thisRStartA = thisR.getStartAttribute();
                            RERAttribute thisREndA = thisR.getStartAttribute();
                            RERAttribute thatRStartA = (isReverse) ? thatR.getEndAttribute() : thatR.getStartAttribute();
                            RERAttribute thatREndA = (isReverse) ? thatR.getStartAttribute() : thatR.getEndAttribute();
                            
                            if (thisRStartA != null && thatRStartA != null &&
                                    thisREndA != null && thatREndA != null) {
                                if (thisRStartA.getName().equals(thatRStartA.getName()) &&
                                        thisREndA.getName().equals(thatREndA.getName())) {
                                    match++;
                                }
                                if (thisRStartA.isPrimaryKey() == thatRStartA.isPrimaryKey() &&
                                        thisREndA.isPrimaryKey() == thatREndA.isPrimaryKey()
                                        ) {
                                    match++;
                                }
                            }
                        }
                        if (match == quality) {
                            thisRelationshipBag.remove(thisR);
                            thatRelationshipBag.remove(thatR);
                            relationshipMap.put(thisR, thatR);
                            break relationshipMapper;
                        }
                    }
                }
            } while (match == quality && thisRelationshipBag.size()  > 0 && thatRelationshipBag.size() > 0);
        }
        
        // Report differences between relationships
        // ----------------------------------------
        
        for (Map.Entry<Relationship, Relationship> entry : relationshipMap.entrySet()) {
            Relationship thisR = entry.getKey();
            Relationship thatR = entry.getValue();
            
            if (entityMap.get(thisR.getStart()) != thatR.getStart() &&
                    entityMap.get(thisR.getStart()) != thatR.getEnd() ||
                    entityMap.get(thisR.getEnd()) != thatR.getStart() &&
                    entityMap.get(thisR.getEnd()) != thatR.getEnd()) {
                differences.add(labels.getFormatted("diffRelationshipEnds", thisR.getStart().getName(), thisR.getEnd().getName(), thatR.getStart().getName(), thatR.getEnd().getName()));
            } else {
                
                boolean isReverse = entityMap.get(thisR.getStart()) == thatR.getEnd();
                
                int thisRStartC = thisR.getStartCardinality();
                int thisREndC = thisR.getEndCardinality();
                int thatRStartC = (isReverse) ? thatR.getEndCardinality() : thatR.getStartCardinality();
                int thatREndC = (isReverse) ? thatR.getStartCardinality() : thatR.getEndCardinality();
                
                if (thisRStartC != thatRStartC && thatRStartC != Relationship.UNSPECIFIED_CARDINALITY ||
                        thisREndC != thatREndC && thatREndC != Relationship.UNSPECIFIED_CARDINALITY
                        ) {
                    differences.add(labels.getFormatted("diffRelationshipCardinality", thisR.getStart().getName(), thisR.getEnd().getName(), thatRStartC, thatREndC));
                }
                RERAttribute thisRStartA = thisR.getStartAttribute();
                RERAttribute thisREndA = thisR.getEndAttribute();
                RERAttribute thatRStartA = (isReverse) ? thatR.getEndAttribute() : thatR.getStartAttribute();
                RERAttribute thatREndA = (isReverse) ? thatR.getStartAttribute() : thatR.getEndAttribute();
                
                if (thisRStartA == null && thatRStartA != null ||
                        thisREndA == null && thatREndA != null) {
                    differences.add(labels.getFormatted("diffRelationshipSpecifyAttributes", thisR.getStart().getName(), thisR.getEnd().getName()));
                }
                if (thisRStartA != null && thatRStartA != null &&
                        thisREndA != null && thatREndA != null) {
                    if (! thisRStartA.getName().equals(thatRStartA.getName())) {
                        differences.add(labels.getFormatted("diffRelationshipStartAttribute", thisR.getStart().getName(), thisR.getEnd().getName(), thatRStartA.getName()));
                    }
                    if (! thisREndA.getName().equals(thatREndA.getName())) {
                        differences.add(labels.getFormatted("diffRelationshipEndAttribute", thisR.getStart().getName(), thisR.getEnd().getName(), thatREndA.getName()));
                    }
                }
                
                String thisRL = thisR.getStartLabel();
                String thatRL = (isReverse) ? thatR.getEndLabel() : thatR.getStartLabel();
                boolean thatRLTraversable = (isReverse) ? thatR.isEndTraversable() : thatR.isStartTraversable();
                if (thatRLTraversable && thatRL != null && (thisRL == null || ! thisRL.equals(thatRL))) {
                    differences.add(labels.getFormatted("diffRelationshipName", thisR.getStart().getName(), thisR.getEnd().getName(), thatRL));
                }
                thisRL = thisR.getEndLabel();
                thatRL = (isReverse) ? thatR.getStartLabel() : thatR.getEndLabel();
                thatRLTraversable = (isReverse) ? thatR.isStartTraversable() : thatR.isEndTraversable();
                if (thatRLTraversable && thatRL != null && (thisRL == null || ! thisRL.equals(thatRL))) {
                    differences.add(labels.getFormatted("diffRelationshipName", thisR.getStart().getName(), thisR.getEnd().getName(), thatRL));
                }
                
                if (thisR.isResolved() != thatR.isResolved()) {
                    differences.add(labels.getFormatted("diffRelationshipResolved", thisR.getStart().getName(), thisR.getEnd().getName(), thatR.isResolved() ? 0 : 1));
                }
            }
        }
        
        
        for (Relationship r : thisRelationshipBag) {
            differences.add(labels.getFormatted("diffRelationshipTooMuch", r.getStart().getName(), r.getEnd().getName()));
        }
        for (Relationship r : thatRelationshipBag) {
            if (! thatEntityBag.contains(r.getStart()) && ! thatEntityBag.contains(r.getEnd())) {
                differences.add(labels.getFormatted("diffRelationshipMissing", r.getStart().getName(), r.getEnd().getName()));
            }
        }
        
        return differences;
    }
}